<?php
/*--------------------------------------------------------------
   GambioVisitorsCount.php 2021-08-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use NumberFormatter;

class GambioVisitorsCount extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.visitors.count';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Besucher',
        self::LANGUAGE_CODE_ENGLISH => 'Visitors'
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var NumberFormatter
     */
    private $numberFormatter;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        Connection $connection,
        NumberFormatter $numberFormatter
    ) {
        $this->dataFactory     = $dataFactory;
        $this->connection      = $connection;
        $this->numberFormatter = $numberFormatter;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createText(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdownIncludingToday($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $timespan = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)->value());
        $visitors = $this->connection->createQueryBuilder()
                        ->select('SUM(gm_counter_visits_total) AS visitors')
                        ->from('gm_counter_visits')
                        ->where('gm_counter_date BETWEEN :startDate AND :endDate')
                        ->setParameters([
                                            ':startDate' => $timespan->startDate()
                                                ->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
                                            ':endDate'   => $timespan->endDate()
                                                ->format(self::QUERY_TIMESPAN_FORMAT_DATE_END)
                                        ])
                        ->execute()
                        ->fetchAll()[0];
        
        return $this->dataFactory->useTextData()->createTextData($this->dataFactory->useTextData()
                                                                     ->createValue($this->numberFormatter->format((int)($visitors['visitors']
                                                                                                                        ??
                                                                                                                        0))));
    }
}
